/**
 * $Author: magi $ $Date: 2000/01/30 22:20:38 $ $Revision: 1.5 $
 *
 * $Log: wwbase.h,v $
 * Revision 1.5  2000/01/30 22:20:38  magi
 * added wwbase.cc for WWObject handling
 *
 * Revision 1.4  2000/01/08 00:50:21  magi
 * Client connect/disconnect works. BUILD works, MOVE works partially.
 *
 * Revision 1.3  2000/01/04 11:28:31  magi
 * *** empty log message ***
 *
 **/

#ifndef __WWBASE_H__
#define __WWBASE_H__

#include <magic/Math.h>
#include <magic/pararr.h>
#include <magic/refarray.h>

class WWWorld;			// In wwworld.h
class WWObjectSpace;	// Internal


class WWCoordInt {
	static const int around_x[8];
	static const int around_y[8];
  public:
				WWCoordInt		() : x(0), y(0) {}
				WWCoordInt		(int X, int Y) : x(X), y(Y) {}
				WWCoordInt		(const WWCoordInt& o) {x=o.x; y=o.y;}
				operator=		(const WWCoordInt& o) {x=o.x; y=o.y;}
	WWCoordInt	around8			(int d) const {return WWCoordInt(x+around_x[d], y+around_y[d]);}

  public:
	int x;
	int y;
};

class WWCoord {
	static const int around_x[8];
	static const int around_y[8];
  public:
					WWCoord				() : x(0.0), y(0.0) {}
					WWCoord				(int X, int Y) : x(X), y(Y) {}
					WWCoord				(double X, double Y) : x(X), y(Y) {}
					WWCoord				(const WWCoord& o) {x=o.x; y=o.y;}
					operator=			(const WWCoord& o) {x=o.x; y=o.y;}
					operator+=			(const WWCoord& o) {x+=o.x; y+=o.y;}
					operator WWCoordInt	() const {return WWCoord (int(x+0.5), int(y+0.5));}
	WWCoordInt		round				() const {return WWCoordInt (x+0.5, y+0.5);}
	
	double			distance			(const WWCoord& o) const {return sqrt(sqr(o.y-y)+sqr(o.x-x));}
	double			distanceSqr			(const WWCoord& o) const {return sqr(o.y-y)+sqr(o.x-x);}

	/** Returns direction to the other point in radians. */
	double			direction			(const WWCoord& o) const {return atan2 (o.y-y, o.x-x);}
	
	WWCoord			around8				(int d) const {return WWCoord(x+around_x[d], y+around_y[d]);}
	
  public:
	double x;
	double y;
};



class Radian {
  public:
					Radian			(double x) : mValue(x)	{}
	
					operator double	() const				{return mValue;}
	Radian&			operator +=		(const Radian& o)		{mValue = toRange(mValue+o.mValue); return *this;}
	Radian			operator +		(double o)				{return toRange(mValue + o);}
	Radian			operator +		(const Radian& o)		{return toRange(mValue + o.mValue);}
	Radian			operator -		(double o)				{return toRange(mValue - o);}
	Radian			operator -		(const Radian& o)		{return toRange(mValue - o.mValue);}
	Radian			round8			() const				{return toRange (int(mValue/M_PI*4+0.5)*M_PI*0.25);}

  private:
	Radian			toRange			() const				{return Radian(fmod(mValue,M_PI*2));}
	static double	toRange			(double x)				{return fmod(x,M_PI*2);}
	
  private:
	double mValue;
};



///////////////////////////////////////////////////////////////////////////////
//                |   | |   |  ___          o                                //
//                | | | | | | |   | |          ___   ___   |                 //
//                | | | | | | |   | |---    | /   ) |   \ -+-                //
//                | | | | | | |   | |   )   | |---  |      |                 //
//                 V V   V V  `___ |__/  \_|  \__   \__/   \                //
///////////////////////////////////////////////////////////////////////////////

class WWObject {
  public:
						WWObject		(WWObject* owner=NULL) : mpOwner(owner) {mId=-1; mpContainer=NULL;}
	virtual				~WWObject		() {}
	
	int					id				() const {return mId;}
	void				removeChildren	();
	virtual void		execute			()=0;

	/** Returns the object that owns the object. */
	WWObject*			owner			() {return mpOwner;}
	const WWObject*		owner			() const {return mpOwner;}

	WWObjectSpace&		container		() {return *mpContainer;}

	WWWorld&			world			();
	
  protected:
	int					mId;
	WWObject*			mpOwner;
	/*RefArray<WWObject>	mChildren;*/
	WWObjectSpace*		mpContainer;

	friend WWObjectSpace;
	void				setId			(int id) {mId=id;}
};



//////////////////////////////////////////////////////////////////////////////
// |   | |   |  ___                                   |             |       //
// | | | | | | /   \                   ___    _       |  ___  |     |  ___  //
// | | | | | | |      __  |/|/| |/|/|  ___| |/ \   ---|  ___| |---  | /   ) //
// | | | | | | |     /  \ | | | | | | (   | |   | (   | (   | |   ) | |---  //
//  V V   V V  \___/ \__/ | | | | | |  \__| |   |  ---|  \__| |__/  |  \__  //
//////////////////////////////////////////////////////////////////////////////

/** Not in use currently. */
class WWCommandable {
  public:
						WWCommandable	() : mpCommander(NULL) {}
	
	/** Returns the commander of the object. */
	WWObject*			commander		() {return mpCommander;}
	const WWObject*		commader		() const {return mpCommander;}

	/** Alters the command chain. */
	void				setCommander	(WWObject* newCommander) {mpCommander=newCommander;}

	/** Tells whether the given object can command this object. */
	bool				canCommand		(const WWObject& obj) const {return false;} 
	
  protected:
	WWObject*			mpCommander;
};



//////////////////////////////////////////////////////////////////////////////
// |   | |   |  ___          o                  ----                        //
// | | | | | | |   | |          ___   ___   |  (      --   ___   ___   ___  //
// | | | | | | |   | |---    | /   ) |   \ -+-  ---  |  )  ___| |   \ /   ) //
// | | | | | | |   | |   )   | |---  |      |      ) |--  (   | |     |---  //
//  V V   V V  `___ |__/  \_|  \__   \__/   \ ___/  |     \__|  \__/  \__  //
//////////////////////////////////////////////////////////////////////////////

class WWObjectSpace {
  public:
						WWObjectSpace	();
	virtual				~WWObjectSpace	() {}
	
	/** Adds the given object to the container.
	 *
	 *  Returns the ID of the object.
	 **/
	virtual int			add				(WWObject* object);

	/** Removes the object from the container.
	 **/
	virtual void		remove			(int objectId);

	/** Returns the object with index i. The result may be NULL, if the
	 *  object id is empty.
	 **/
	WWObject*			object			(int objectId) {return mObjects.getp(objectId);}

	/** Runs the objects. */
	virtual void		executeObjects	();

	/** Returns the number of object IDs, i.e., the largest object Id
	 *  number + 1.
	 **/
	int					objectIds		() const {return mObjects.size;}

	WWWorld&			world			() {return *mpWorld;}

  protected:
	/** All owned objects */
	Array<WWObject>	mObjects;

	WWWorld*		mpWorld;
};

#endif
