/**
 * $Author: magi $ $Date: 2000/01/30 22:20:39 $ $Revision: 1.7 $
 *
 * $Log: wwunit.h,v $
 * Revision 1.7  2000/01/30 22:20:39  magi
 * added wwbase.cc for WWObject handling
 *
 * Revision 1.6  2000/01/08 00:50:21  magi
 * Client connect/disconnect works. BUILD works, MOVE works partially.
 *
 * Revision 1.5  2000/01/04 11:28:31  magi
 * *** empty log message ***
 *
 **/

#ifndef __WWUNIT_H__
#define __WWUNIT_H__

#include <magic/cstring.h>
#include <magic/map.h>
#include "wwbase.h"

class WWPlayer;		// In wwplayer.h
class UnitHandler;	// In wwworld.h
class WWWorld;		// In wwworld.h
class WWUnitState;	// Internal
class WWUnitInst;	// Internal

///////////////////////////////////////////////////////////////////////////////
//  |   | |   | |   |                              ___  |                    //
//  | | | | | | | | |  ___   ___   --         _   /   \ |  ___   ____  ____  //
//  | | | | | | | | | /   )  ___| |  )  __  |/ \  |     |  ___| (     (      //
//  | | | | | | | | | |---  (   | |--  /  \ |   | |     | (   |  \__   \__   //
//   V V   V V   V V   \__   \__| |    \__/ |   | \___/ |  \__| ____) ____)  //
///////////////////////////////////////////////////////////////////////////////

class WWWeaponClass {
  public:
					WWWeaponClass	(const String& name, const StringMap& params);

	enum	ammoClasses	{AC_BEAM, AC_PROJECTILE, AC_MISSILE};
	
  protected:
	String	mName;
	double	mRange;
	double	mEnergy;
	int		mAmmoClass;
};

///////////////////////////////////////////////////////////////////////////////
//          |   | |   | |   |       o      ___  |                            //
//          | | | | | | |   |   _      |  /   \ |  ___   ____  ____          //
//          | | | | | | |   | |/ \  | -+- |     |  ___| (     (              //
//          | | | | | | |   | |   | |  |  |     | (   |  \__   \__           //
//           V V   V V  `___ |   | |   \ \___/ |  \__| ____) ____)          //
///////////////////////////////////////////////////////////////////////////////

/** Unit definition.
 **/
class WWUnitClass {
  public:
	/** Default constructor, usage FORBIDDEN. */
					WWUnitClass		() {FORBIDDEN}

					WWUnitClass		(const String& name, const StringMap& params);

	/** Creates a new instance of the class.
	 **/
	WWUnitInst*		newUnit			(WWPlayer* owner) const;

	int				id				() const {return mId;}
	const String&	name			() const {return mName;}
	const String&	weaponName		() const {return mWeaponName;}
	double			maxhits			() const {return mMaxHits;}
	double			speed			() const {return mSpeed;}
	double			rotspeed		() const {return mRotSpeed;}
	double			cost			() const {return mResourceCost;}
	double			buildTime		() const {return mBuildTime;}
	double			buildEfficiency	() const {return mBuildEffic;}
	const String&	builds			() const {return mBuilds;}
	const WWWeaponClass* weapon		() const {return mWeapon;}
	
  protected:
	int				mId;			// Identifier
	String			mName;			// Name of the unit
	String			mWeaponName;	// Name of the weapon of the unit
	WWWeaponClass*	mWeapon;		// Weapon class
	double			mArmour;		// Amount of armor
	double			mMaxHits;		// Maximum hits
	double			mSpeed;			// Maximum movement speed
	double			mAcceleration;	// Acceleration/deceleration rate m/s^2
	double			mRotSpeed;		// Rotation speed deg/s
	double			mTurretRotSpd;	// Turret rotation speed deg/s
	double			mResourceCost;	// Cost to build
	double			mBuildTime;		// Time to build
	String			mBuilds;		// What the unit can build; space separated list
	double			mBuildEffic;	// Efficiency in building other units
};



///////////////////////////////////////////////////////////////////////////////
//                |   | |   | |   |       o     |     o                      //
//                | | | | | | |   |   _      |  |       |                    //
//                | | | | | | |   | |/ \  | -+- |     | |---                 //
//                | | | | | | |   | |   | |  |  |     | |   )                //
//                 V V   V V  `___ |   | |   \ |____ | |__/                 //
///////////////////////////////////////////////////////////////////////////////

/** Contains all the availabe unit classes.
 **/
class WWUnitLib {
  public:
							WWUnitLib	() {}

	/** Constructs the units according to the parameters.
	 **/
	void					make		(const StringMap& params);

	/** Returns the global UnitLib singleton.
	 **/
	static WWUnitLib&		getInst		() {return gUnitLib;}

	/** Returns a new unit of the given unitclass.
	 **/
	static WWUnitInst*		newUnit		(const String& name, WWPlayer* player);

	static const WWUnitClass* unitClass	(const String& name);

	static const WWUnitClass* unitClass	(int classid);
	
  protected:
	Array<WWUnitClass>		mUnitClasses;
	Array<WWWeaponClass>	mWeaponClasses;

	static WWUnitLib		gUnitLib;
};


//////////////////////////////////////////////////////////////////////////////
//           |   | |   |  ___                                   |           //
//           | | | | | | /   \                   ___    _       |           //
//           | | | | | | |      __  |/|/| |/|/|  ___| |/ \   ---|           //
//           | | | | | | |     /  \ | | | | | | (   | |   | (   |           //
//            V V   V V  \___/ \__/ | | | | | |  \__| |   |  ---|           //
//////////////////////////////////////////////////////////////////////////////

/** Abstract superclass for all unit commands.
 *
 *  The actions of the combat units are executed by command objects
 *  that manipulate the combat units.
 *
 *  The class is not actually abstract because of certain reasons, so
 *  don't instantiate it.
 **/
class WWCommand {
  public:
							WWCommand	() {}

	/** Executes the command, and manipulates the unit according to
	 *  it. Inheritors MUST overload this virtual method.
	 *
	 *  Returns true if the command has finished for good, and may
	 *  thus be destroyed from the command queue of the unit.
	 **/
	virtual bool			execute		(WWUnitInst& unit, WWWorld& world) {MUST_OVERLOAD}

  protected: // Methods
	/** Tells if the execution of the command has been finished.
	 **/
	bool					finished	() const {return mTimes!=0;}

  protected: // Member variables
	/** How many times the command should be repeated more? -1 means
	 *  infinitely.
	 **/
	int		mTimes;

};

/** Unit command that moves the unit to a location.
 **/
class WWComMove : public WWCommand {
  public:
							WWComMove	(const WWCoord& target) : mTarget(target) {}

	virtual bool			execute		(WWUnitInst& unit, WWWorld& world);
	
  protected:
	/** Extrapolates the unit's position into future. */
	WWUnitState				extrapolate (WWUnitInst& unit, double dt) const;
	
  protected:
	WWCoord		mTarget;
};

/** Unit command that fires the weapon at a location.
 **/
class WWComFire : public WWCommand {
  public:
							WWComFire	(const WWCoord& target) : mTarget(target) {}
	
	virtual bool			execute		(WWUnitInst& unit, WWWorld& world);

  protected:
	WWCoord		mTarget;
};

/** Unit command that builds a specific unit.
 **/
class WWComBuild : public WWCommand {
  public:
	/** Constructor.
	 *
	 * @param uclass Unit class of the unit to be built.
	 *
	 * @param player Player who builds the unit.
	 **/
	   
							WWComBuild	(const WWUnitClass* uclass, WWPlayer* player)
									: mpClass(uclass), mpUnit(NULL), mpPlayer(player) {}
							~WWComBuild	();
	
	virtual bool			execute		(WWUnitInst& builder, WWWorld& world);
	
  protected:
	const WWUnitClass*	mpClass;
	WWUnitInst*			mpUnit; 	/** The incomplete unit. */
	WWPlayer*			mpPlayer;
};



///////////////////////////////////////////////////////////////////////////////
//          |   | |   | |   |       o      ----                              //
//          | | | | | | |   |   _      |  (      |   ___   |   ___           //
//          | | | | | | |   | |/ \  | -+-  ---  -+-  ___| -+- /   )          //
//          | | | | | | |   | |   | |  |      )  |  (   |  |  |---           //
//           V V   V V  `___ |   | |   \ ___/    \  \__|   \  \__           //
///////////////////////////////////////////////////////////////////////////////

class WWUnitState {
  public:
							WWUnitState	() : mLocation(-1,-1) {}
							WWUnitState (const WWUnitState& other) {operator=(other);}

							operator=	(const WWUnitState& other);
	
	/** Moves the unit instantly to the target position.
	 *
	 *  Returns NULL if the coordinates of the position are invalid,
	 *  or the target location is alread occupied. The unit's location
	 *  will remain unmodified. If the move is legal, returns pointer
	 *  to self.
	 **/
	virtual WWUnitState*	moveTo			(const WWCoord& pos);

	/** Returns non-zero value if the unit would collide with
	 *  something if it were located at the given coordinates.
	 *
	 *  Return values can have following bit flags:
	 *      WWUICOLL_UNIT    (another unit),
	 *      WWUICOLL_TERRAIN (a terrain feature),
	 *      WWUICOLL_BORDER  (map border).
	 **/
	virtual unsigned int	collision		(const WWCoord& pos, WWWorld& world) const;

	/** Returns the current coordinates of the unit. */
	const WWCoord&			location		() const {return mLocation;}

	/** Returns the current orientation of the unit in radians. */
	double					direction		() const {return mDirection;}

	/** Turn the unit instantly to new direction. */
	virtual double			turn			(double direction) {mDirection=direction;}

  protected:
	WWCoord				mLocation;		/* Coordinates on the map. */
	double				mDirection;		/* Current driving direction. */
	double				mTurretDir;		/* Current turret direction. */
};



///////////////////////////////////////////////////////////////////////////////
//             |   | |   | |   |       o     ---                             //
//             | | | | | | |   |   _      |   |    _    ____  |              //
//             | | | | | | |   | |/ \  | -+-  |  |/ \  (     -+-             //
//             | | | | | | |   | |   | |  |   |  |   |  \__   |              //
//              V V   V V  `___ |   | |   \ _|_ |   | ____)   \             //
///////////////////////////////////////////////////////////////////////////////

/** Unit instance class.
 **/
class WWUnitInst : public WWObject, public WWUnitState {
  public:
							WWUnitInst	 	(WWWorld& world, WWObject* owner=NULL) {FORBIDDEN}
							WWUnitInst		(const WWUnitClass& uclass, WWObject* player);

	/*virtual*/ WWUnitInst*	moveTo			(const WWCoord& pos);

	/*virtual*/ double		turn			(double direction) {mDirection=direction; mMoved=true;}

	
	/** Returns a flag that indicated whether or not the unit has
	 *  already moved during this game cycle.
	 **/
	bool					hasMoved		() const {return mMoved;}

	/** Clears the flag indicating whether or not the unit has already
	 *  moved during this game cycle.
	 **/
	void					clearMoved		() {mMoved=false;}

	/** Percentage of hitpoints remaining. */
	double					hitpoints100	() {return 100*mHitpoints/mpUnitClass->maxhits();}

	/** Makes damage to the unit. Returns true iff it is
	 *  destroyed (negative hitpoints).
	 **/
	bool					addDamage		(double damage);

	/** Heals the unit. Returns true iff it is at full
	 *  hitpoints.
	 **/
	bool					heal			(double hitpoints);

	/** Builds the unit. Returns true iff it is at full
	 *  hitpoints.
	 **/
	bool					build			(double amount=0.0);
	
	/** Returns the class object for the instance.
	 **/
	const WWUnitClass&		getClass	() const {return *mpUnitClass;}

	/** Adds a command to the command queue.
	 **/
	void					addCommand		(WWCommand* cmd, bool clear=false) {if (clear) removeCommand(true); mCommands.add (cmd);}

	/** Reads the first command. */
	WWCommand*				currentCommand	() {return (mCommands.size>0)? &mCommands[0] : NULL;}

	/** Removes the first command. */
	void					removeCommand	(bool all=false) {if (all) mCommands.make(0); else mCommands.removeFill(0);}
	
	/** Executes the current command, if any. */
	void					execute			();

	/** Sets the number of search mode cycles left. */
	void					searchMode		(int cycles) {mSearchMode = cycles;}
	
	/** Returns the number of search mode cycles left. */
	int						searchMode		() const {return mSearchMode;}

  protected:
	const WWUnitClass*	mpUnitClass;	/* Unit's class. */
	Array<WWCommand>	mCommands;		/* Command queue. */
	double				mHitpoints;		/* Current hitpoints. */
	double				mBuildComplete;	/* Progress in building this unit. */
	bool				mMoved;			/* Has the unit moved or turned in this cycle? */
	int					mSearchMode;
	int					mId;
};

#define WWUICOLL_UNIT		0x0001
#define WWUICOLL_TERRAIN	0x0002
#define WWUICOLL_BORDER		0x0004

#endif
