/**
 * $Author: magi $ $Date: 2000/01/31 10:21:54 $ $Revision: 1.15 $
 *
 * $Log: wwserver.cc,v $
 * Revision 1.15  2000/01/31 10:21:54  magi
 * Uses now the revised global objectid system
 *
 * Revision 1.14  2000/01/08 00:50:22  magi
 * Client connect/disconnect works. BUILD works, MOVE works partially.
 **/

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>

#include <magic/pararr.h>
#include <magic/applic.h>
#include <magic/Math.h>

#include "wwserver.h"
#include "config.h"


///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//        ----                           ----  |                             //
//       (      ___             ___      |   ) |  ___         ___            //
//        ---  /   ) |/\ |   | /   ) |/\ |---  |  ___| \   | /   ) |/\       //
//           ) |---  |    \ /  |---  |   |     | (   |  \  | |---  |         //
//       ___/   \__  |     V    \__  |   |     |  \__|   \_/  \__  |         //
//                                                      \_/                  //
///////////////////////////////////////////////////////////////////////////////

ServerPlayer::ServerPlayer (WWWorld* owner, int socket, const sockaddr_in& address)
		: WWPlayer () {
	mSocket = socket;
	memcpy (&mAddress, &address, sizeof (address));

	// Set the socket as non-blocking
	int flags = fcntl (mSocket, F_GETFL, 0);
	fcntl (mSocket, F_SETFL, O_NONBLOCK | flags);
}

void ServerPlayer::launch () {
	addMsg ("LCH\n");
	
	changeRes (+1000.0); // Add 1000 resource points
	
	// Add a TankFactory as the initial unit
	WWUnitInst* factory = WWUnitLib::newUnit("TankFactory", this);
	int id = container().add (factory);

	// Move it to a random location
	WWMap& map = world().map();
	WWCoord startpos;
	do { // Find a free random location on the map
		startpos = WWCoord (rnd(map.sizex()-1)+0.5, rnd(map.sizey()-1)+0.5);
		printf ("Trying factory location %f,%f\n", startpos.x,startpos.y);
	} while (!factory->moveTo (startpos));
	printf ("TankFactory %d at %f,%f (%d,%d)\n", id, startpos.x, startpos.y,
			map.sizex(), map.sizey());

	addMsg (world().scoreMsg());

}

ServerPlayer::~ServerPlayer () {
	close (mSocket);
}

/*
int ServerPlayer::addUnit (WWUnitInst* unit) {
	int unitid = WWPlayer::addUnit (unit);

	// Report the creation to all the players
	String msg = format("UNT %d %d %s\n", id(), unitid,
						(CONSTR) unit->getClass().name());
	static_cast<ServerWorld&>(world()).addPublicMsg (msg);
	return unitid;
}
*/

bool ServerPlayer::changeRes (double amount) {
	bool result = WWPlayer::changeRes (amount);
	
	// Report the creation to all the players
	addMsg (format("RES %f\n", resources()));

	return result;
}

String ServerPlayer::ip () const {
	return format ("%d.%d.%d.%d",
				   (mAddress.sin_addr.s_addr>>24) & 255,
				   (mAddress.sin_addr.s_addr>>16) & 255,
				   (mAddress.sin_addr.s_addr>>8) & 255,
				   mAddress.sin_addr.s_addr & 255);
}

/*virtual*/ void ServerPlayer::unitHit (WWUnitInst& unit) {
	addMsg (format("HIT %d %d\n", unit.id(), int(unit.hitpoints100())));
}

ServerWorld& ServerPlayer::world () {
	return dynamic_cast<ServerWorld&>(mpContainer->world());
}


//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//           ----                           |   |          |     |          //
//          (      ___             ___      | | |          |     |          //
//           ---  /   ) |/\ |   | /   ) |/\ | | |  __  |/\ |  ---|          //
//              ) |---  |    \ /  |---  |   | | | /  \ |   | (   |          //
//          ___/   \__  |     V    \__  |    V V  \__/ |   |  ---|          //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

ServerWorld::ServerWorld (const StringMap& params) {
	mSocket = 0;
	mTime = 0;
	mPortNumber = params["ServerWorld.port"];
	mRunning = false;

	// Create unit library
	WWUnitLib& unitlib = WWUnitLib::getInst ();
	unitlib.make (params);
}

void ServerWorld::createSocket () {
	// Create socket
	mSocket = socket (AF_INET, SOCK_STREAM, 0);
	if (mSocket == -1)
		throw Exception (format ("Could not create a socket: %s\n", strerror(errno)));

	// Force it to open even if already in use
	int one = 1;
	setsockopt(mSocket, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(int));
	
	// Bind it
	mAddress.sin_family = AF_INET;
	mAddress.sin_port = htons(mPortNumber);
	mAddress.sin_addr.s_addr = htonl(INADDR_ANY);
	if (bind (mSocket, (sockaddr*) &mAddress, sizeof (mAddress)) == -1)
		throw Exception (format ("Could not bind a socket: %s", strerror(errno)));

	// Create socket queue
	int squeue = listen (mSocket, 10);

	// Set socket as non-blocking
	int flags = fcntl (mSocket, F_GETFL, 0);
	fcntl (mSocket, F_SETFL, O_NONBLOCK | flags);
}

void ServerWorld::createPlayer (int asock, const sockaddr_in& clientAddress) {
	ServerPlayer* newPlayer = new ServerPlayer (this, asock, clientAddress);
	// Tell other players about the new player. The VERSION is defined
	// in config.h.
	newPlayer->addMsg (format ("VER %s\n", VERSION));

	int id = add (newPlayer);
	printf ("Accepted player %d from %s\n", id, (CONSTR) newPlayer->ip());
	
	// Tell the new player about the other players
	for (int i=0; i<16; i++)
		if (i!=id && object(i)) {
			ServerPlayer& player = dynamic_cast<ServerPlayer&> (*object(i));
			newPlayer->addMsg (format("PLR %d %s %s\n", i, (CONSTR) player.ip(),
									  (CONSTR) player.name()));
			// Tell other players about the new player
			//player.addMsg (format ("PLR %d %s Player%d\n", id, (CONSTR) newPlayer->ip(), id));
		}

	if (mRunning) {
		newPlayer->launch ();

		// Send unit data to the new player
		for (int u=0; u<objectIds(); u++)
			if (WWUnitInst* unit = dynamic_cast<WWUnitInst*>(object(u))) {
				newPlayer->addMsg (newMsg (*unit));
				newPlayer->addMsg (posMsg (*unit));
			}
	}

}

/*virtual*/ int ServerWorld::add (WWObject* object) {
	int id = WWObjectSpace::add (object);
	if (WWUnitInst* unit = dynamic_cast<WWUnitInst*>(object)) {
		addPublicMsg (newMsg (*unit));
		addPublicMsg (format ("POS %d 100 100 0\n", id));
		mNewUnit = id; // KLUDGE
	}
	return id;
}

/*virtual*/ void ServerWorld::remove (int objectId) {

	if (dynamic_cast<WWPlayer*>(object(objectId))) {
		// The client disconnects -> remove the player object
		addPublicMsg (format ("QIT %d\n", objectId));
		printf ("Disconnecting player %d\n", objectId);
		
		// Check if there are any players left
		if (mRunning) {
			mRunning = false; // If not, go back to launch mode
			for (int p=0; p<16; p++)
				if (object(p) && dynamic_cast<WWPlayer*>(object(p)))
					mRunning=true;
		}
	} else if (dynamic_cast<WWUnitInst*>(object(objectId))) {
		// Removing a combat unit
		//addPublicMsg (format ("DSY %d 1\n", objectId));
		addPublicMsg (format ("HIT %d 0\n", objectId));
		addPublicMsg (scoreMsg());
	}

	WWWorld::remove (objectId);
}

void ServerWorld::run () {
	ASSERTWITH (!mSocket, "ServerWorld socket already created - error");
	createSocket ();
	
	// Start listening
	sockaddr_in clientAddress;
	bool quitloop = false;
	while (!quitloop) {
		unsigned int alen = sizeof (clientAddress);

		clearPublicMsgs(); // Clear broadcast message buffer
		if (mRunning)
			addPublicMsg ("INI\n");
		
		// Check for new connections
		int asock = accept (mSocket, (sockaddr*) &clientAddress, &alen);
		if (asock >= 0)
			FAILTRACE (createPlayer (asock, clientAddress));

		// Check if there's data from any of the sockets
		recvFromClients ();

		// Run the world - move, create and destroy units, and fire weapons.
		executeObjects ();
		
		// Send message buffers to the clients
		sendToClients ();

		//synchronizeClients ();

		// Wait for the next game tick
		usleep (int(ticTime()*1000000));

		mTime++;
	}

}

void ServerWorld::recvFromClients () {
	char buffer [1024];
	for (int i=0; i<16; i++)
		if (object(i)) {
			ServerPlayer& player = dynamic_cast<ServerPlayer&> (*object(i));
			
			int rsize = recv (player.getSocket(), buffer, 1024, MSG_NOSIGNAL);

			// Check if the player has hung up. (rsize-1 means some
			// other error, such as empty receive).
			if (rsize==0) {
				printf ("Player %d hung up: %s\n", i, strerror(errno));
				addPublicMsg (format ("QIT %d\n", i));
				remove (i); // Remove the player object
				break;
			}
			
			if (rsize>0) {
				// Some data had arrived, a command
				buffer[rsize] = 0;
				String buffstr = buffer;

				for (int compos=0; compos<buffstr.len; ) {
					int pos = buffstr.find ('\n', compos);
					
					if (pos!=-1) {
						String command = buffstr.substr (compos, pos-compos);
						command.chop ();
						printf ("Client %d: %s\n", i, (CONSTR) command);
						Array<String> params;
						command.split (params, ' ');
						params[0].uppercase ();
						
						handleCommand (player, params);
					} else
						break;

					compos = pos+1;
				}
			}
		}
}

void ServerWorld::handleCommand (ServerPlayer& player, const Array<String>& params) {
	String command = params[0];
	
	if (command == "QIT") {
		remove (player.id());
		
	} else if (command == "PLR" && params.size==2) {
		player.setName (params[1]);
		printf ("Player %d name %s\n", player.id(), (CONSTR) player.name());
		addPublicMsg(format ("PLR %d %s %s\n", player.id(), (CONSTR) player.ip(), (CONSTR) player.name()));
		
	} else if (command == "LCH" && params.size==1 && !mRunning) {
		if (player.id()==0) {
			printf ("Game was launched\n");
			for (int p=0; p<16; p++)
				if (ServerPlayer* aplayer = dynamic_cast<ServerPlayer*>(object(p)))
					aplayer->launch ();
			mRunning = true;
			printf ("Player %d launched\n", player.id());
		} else
			player.addMsg ("ERR 1 Launch failed - only the first joining player can launch the game.\n");
		
	} else if (command == "BLD" && params.size==4) {
		//int unitid = int(params[1]);
		// TEMPORARY WORKAROUND:
		// Find first factory belonging to this player
		int factoryid = -1;
		for (int i=0; i<objectIds(); i++)
			if (WWUnitInst* unit = dynamic_cast<WWUnitInst*>(object(i)))
				if (unit->getClass().name()=="TankFactory" && unit->owner()->id() == player.id()) {
					factoryid = i;
					printf ("Factory %d allocated for the task\n", i);
					break;
				}

		if (factoryid != -1) {
			int unittype = params[1];
			if (const WWUnitClass* uclass = WWUnitLib::unitClass(unittype)) {
				printf ("Player %d wants unit %d to start building %s\n",
						player.id(), factoryid, (CONSTR)uclass->name());
				dynamic_cast<WWUnitInst*>(object(factoryid))->addCommand (new WWComBuild (uclass, &player), true);
			} else
				printf ("ERROR: NO SUCH UNIT TYPE %s FOUND!\n", (CONSTR) unittype);
		} else
			printf ("Could not find factory for player %d\n", player.id());
		
	} else if (command == "SEL" && params.size>1) {
		PackArray<int> selection (params.size-1);
		int u=0;
		// Relay the aimin data to all the other players
		String selmsg = format ("SEL %d", player.id());
		for (int p=1; p<params.size; p++)
			if (WWUnitInst* unit = dynamic_cast<WWUnitInst*>(object(int(params[p]))))
				if (unit->owner()->id() == player.id()) { // Allow selection of only units owned by the player
					selection[u++] = unit->id();
					selmsg += format (" %d", unit->id());
				}
		selmsg += '\n';
		addPublicMsg (selmsg);
		selection.resize (u);
		player.select (selection);
		
	} else if (command == "AIM" && params.size==3) {
		double x = int(params[1])/32.0-0.5;
		double y = int(params[2])/32.0-0.5;
		WWCoord location (x, y);
		player.aimAt (location);
		// Relay the aimin data to all the other players
		addPublicMsg (format ("AIM %d %d %d\n", player.id(),
							  int(32*(x+0.5)), int(32*(y+0.5))));
		
	} else if (command == "MOV" && params.size==2) {
		// Put references to the selected units to a reference array
		RefArray<WWUnitInst> sel (player.selection().size);
		for (int i=0; i<player.selection().size; i++)
			if (WWUnitInst* unit = dynamic_cast<WWUnitInst*>(object(player.selection()[i])))
				sel.put (unit, i);

		// Calculate target positions for the units according to the formation
		PackArray<WWCoord> trgCoords = mFormationLib.formation(0).form (sel, player.aim(), 0.0);

		// Command the units
		for (int i=0; i<player.selection().size; i++)
			dynamic_cast<WWUnitInst*>(object(player.selection()[i]))->addCommand (new WWComMove (trgCoords[i]), true);
		
	} else if (command == "SHT" && params.size==3) {
		printf ("%d units selected for shooting.\n");
		for (int i=0; i<player.selection().size; i++)
			if (WWUnitInst* unit = dynamic_cast<WWUnitInst*>(object(player.selection()[i]))) {
				printf ("Unit %d commanded to shoot at %f,%f\n", i, player.aim().x, player.aim().y);
				unit->addCommand (new WWComFire (player.aim()), true);
			}
		
	} else if (command == "MAP") { // Undocumented server command for displaying map
		WWMap& wwmap = map ();
		String mapstr;
		mapstr.reserve (wwmap.sizey()*(wwmap.sizex()+1));
		for (int row=0; row<wwmap.sizey(); row++) {
			for (int col=0; col<wwmap.sizex(); col++) {
				if (WWUnitInst* unit = wwmap.getTerrain (WWCoord (col,row)).unit())
					if (unit->getClass().name()=="Tank")
						printf ("T");
					else
						printf ("F");
				else
					printf (".");
			}
			printf ("\n");
		}
		player.addMsg (mapstr);
	}
}

void ServerWorld::sendToClients () {
	// Send unit locations to the clients
	for (int u=0; u<objectIds(); u++)
		if (WWUnitInst* unit = dynamic_cast<WWUnitInst*> (object(u)))
			if (unit->hasMoved()) {
				addPublicMsg (posMsg (*unit));
				unit->clearMoved ();
			}

	// Send data to the clients
	for (int i=0; i<16; i++)
		if (object(i)) {
			ServerPlayer& player = dynamic_cast<ServerPlayer&> (*object(i));

			String msg = player.msgBuffer();
			//msg += mPublicMsgs;
			if (mRunning)
				msg += "END\n";
			
			// Send any public messages to the player
			send (player.getSocket(), msg, msg.len, 0);
			player.clearMsgs ();
		}
			
}

/*virtual*/ ServerWorld::shootAt (const WWUnitInst& unit, const WWCoord& target) {
	addPublicMsg (format ("SHT %d %d %d\n", unit.id(),
						  int(target.x*32+16), int(target.y*32+16)));
}


///////////////////////////////////////////////////////////////////////////////

void ServerWorld::addPublicMsg (const String& msg) {
	for (int i=0; i<16; i++)
		if (object(i)) {
			ServerPlayer& player = dynamic_cast<ServerPlayer&> (*object(i));
			player.addMsg (msg);
		}
}

String ServerWorld::newMsg (const WWUnitInst& unit) {
	return format ("NEW %d %d %d\n",
				   unit.owner()->id(), unit.id(), (CONSTR) unit.getClass().id());
}

String ServerWorld::posMsg (const WWUnitInst& unit) {
	return format ("POS %d %d %d %d\n",
				   unit.id(), int(unit.location().x*32+16), int(unit.location().y*32+16),
				   (int((1-unit.direction()/(2*M_PI))*255+65536)%256));
}

String ServerWorld::scoreMsg () return result {
	result = "SCR";
	for (int i=0; i<16; i++)
		if (WWPlayer* player = dynamic_cast<WWPlayer*>(object(i))) {
			result += format(" %d %d", /*player->id(), */
							 player->kills(), player->deaths());
		}
	result += '\n';
}


///////////////////////////////////////////////////////////////////////////////
//                            |   |       o                                  //
//                            |\ /|  ___      _                              //
//                            | V |  ___| | |/ \                             //
//                            | | | (   | | |   |                            //
//                            |   |  \__| | |   |                            //
///////////////////////////////////////////////////////////////////////////////

Main() {
	StringMap currentConfig = mParamMap;
	printf ("Server listening to port %s\n",
			(CONSTR) paramMap()["ServerWorld.port"]);
	readConfig ("units.odf");
	mParamMap += currentConfig;
	printf ("Unit classes: %s\n", (CONSTR) paramMap()["WWorld.units"]);
	mParamMap.failByThrow (false);
	
	ServerWorld world (paramMap());

	try {
		printf ("Initializing world...\n");
		world.make (mParamMap["ServerWorld.xsize"], mParamMap["ServerWorld.ysize"]);
		
		printf ("world %dx%d\n", world.map().sizex(), world.map().sizey());
		printf ("Starting game...\n");
		world.run ();

	} catch (Exception& e) {
		fprintf (stderr, "Exception caught at main level:\n%s!\n", (CONSTR) e.what());
	}

	printf ("Game ended, destroying the world...\n");
	printf ("Exiting...\n");
}
