#ifndef __WWBASE_CC_INCLUDED__
#define __WWBASE_CC_INCLUDED__

#define ww_FATAL_ERROR(x) { printf(x); exit(1); }

////////////////////////////////////////////////////////////////////////
// Class:      C_DrawSurface
// Purpose:    A 'draw surface' is a canvas on which you can draw.
//             In practice it's a bitmap that serves as an offscreen
//             buffer to avoid flickering when moving animated objects.
// Inherits:   QPixmap  
////////////////////////////////////////////////////////////////////////
#include <string.h>
#include <qpixmap.h>
class C_DrawSurface : public QPixmap {
    protected:
    public:
};
////////////////////////////////////////////////////////////////////////
// Class:     C_Bitmap
// Purpose:   Provide a wrapper for the Qt's QPixmap class.
//            Maybe even add some new features as needed.
////////////////////////////////////////////////////////////////////////
#include <qpixmap.h>
class C_Bitmap : public QPixmap {
    protected:
    public:
    void drawAt ( C_DrawSurface * surf, short x, short y );
    C_Bitmap ( int w, int h, int depth ) : QPixmap(w, h, depth) {
    }
    C_Bitmap() {
    }
};
////////////////////////////////////////////////////////////////////////
// Class:     C_Frame
// Purpose:   A frame is a bitmap (which in turn is a QPixmap) which
//            also contains hotspot information to allow fine tuning
//            of bitmap placement during animation. Without these
//            offsets all frames must be correctly aligned when drawn.
//            (Which usually isn't that much of a problem) 
////////////////////////////////////////////////////////////////////////
class C_Frame : public C_Bitmap {
    protected:
    short      x_offset, y_offset;
   
    public:
    void paste ( C_DrawSurface *surf, short x, short y ) {
       bitBlt(surf, x-x_offset, y-y_offset, this, 0, 0, -1, -1, QWidget::CopyROP, FALSE);
    }   
    void setXOffset ( short xo ) { x_offset = xo; };
    void setYOffset ( short yo ) { y_offset = yo; };
    void setXYOffsets ( short xo, short yo ) {
       setXOffset(xo); 
       setYOffset(yo);
    }
    C_Frame () {
       setXYOffsets(0,0);
    }
    C_Frame ( short xo, short yo ) {
       setXYOffsets(xo, yo);
    }
    bool loadFrame ( char *file ) {
       if (load(QString(file), NULL, ColorOnly | DiffuseDither | ThresholdAlphaDither)) {
          setMask(createHeuristicMask());      //clipTight=TRUE
	  setOptimization(QPixmap::BestOptim);
	  setXYOffsets(width()/2, height()/2);
	  return TRUE;
       } 
       return FALSE;
    }
    short xOffset() { return (x_offset); };
    short yOffset() { return (y_offset); };
};
////////////////////////////////////////////////////////////////////////
// Class:     C_FramePool
// Purpose:   Stores a collection of frames. In practice, a pool may
//            consist of all frames (=bitmaps) used in the program.
//            That means no relation between them has to be imposed. 
////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <string.h>
class C_FramePool {
    protected:
    C_Frame  *frames;
    short     num_frames;
    public:
    void setNumFrames ( short num ) { num_frames = num; };
    bool allocFrames ( short num ) {
       if (num > 0) {
	  frames = new C_Frame[num];
	  if (frames) {
	     setNumFrames(num);
	     return TRUE;
	  }
       } else ww_FATAL_ERROR("C_FramePool::allocFrames(short) failed! (num <= 0)!\n");
       return FALSE;
    }
    bool incrementalLoadFramesAt ( short base, short count, char *body, char *ext ) {
       char filename[64];
       char number[10];
       if (base + count > num_frames) return FALSE;
       if (base < 0) ww_FATAL_ERROR("C_FramePool::incrementalLoadFramesAt() failed!\n");
       for (int i=0 ; i<count ; i++) {
	  strcpy(filename, body);
	  sprintf(number, "%i", i+1);
	  strcat(filename, number);
	  strcat(filename, ext);
	  if (!frames[base + i].loadFrame(filename))
	    return FALSE;
       }
       return TRUE;
    }
   
   
    void setFrames ( C_Frame *frms ) { frames = frms; };
    bool      isValidFrame ( short frame ) {
	return ((frame < num_frames) & (frame >= 0));
    } 
    C_Frame * getFrame ( short frame ) {
	if ((isValidFrame(frame)) && (frames)) {
	    return (&frames[frame]);
	}
	return NULL;
    }    
    C_FramePool() { 
       setFrames(NULL); 
       setNumFrames(0);
    }
    C_FramePool( C_Frame *frames_, short numframes_ ) {
       setFrames(frames_); 
       setNumFrames(numframes_);
    }   
    C_FramePool( short num ) {
       frames = new C_Frame[num];
       if (!frames) printf("C_FramePool(short) failed!\n");
    }
};


////////////////////////////////////////////////////////////////////////
// Class:     C_FrameList
// Purpose:   The purpose is to index a frame pool (=C_FramePool)
//            In other words, this class provides services to define
//            streams of frames to maintain an animation or a part of
//            an animation.
////////////////////////////////////////////////////////////////////////
class C_FrameList {
    protected:
    short               num_frames;
    short              *frame_number_array;
    C_FramePool        *frame_pool;
    short               max_width, max_height;
   
    public:
    short numFrames() { return(num_frames); };
    bool isValidFrame ( short frame ) { return ((frame >= 0) & (frame < num_frames)) ? 1:0; };
    C_Frame * getFrame ( short frame ) {
	if ((isValidFrame(frame)) && (frame_number_array) && (frame_pool)) {
            return frame_pool->getFrame(frame_number_array[frame]);	    
	}
	return NULL;
    }
    void setFramePool ( C_FramePool *pool ) { frame_pool = pool; };
    void setFrameNumberArray ( short * array ) { frame_number_array = array; };
    void setNumFrames ( short frames ) { num_frames = frames; };
    void setFrameNumberAt ( short index, short fn ) {
       if ((index >= 0) && (index < num_frames)) {
	  frame_number_array[index] = fn;
       } else ww_FATAL_ERROR("C_FrameList::setFrameNumberAt() failed! Array index out of bounds!\n");
    }
    short *frameNumberArray() {  return (frame_number_array); };
    void getMaxBounds ( short &max_w, short &max_h ) {
       int w, h;
       int frame;
       if (max_width > 0) {
	  max_w = max_width;
          max_h = max_height;
	  return;
       }
       max_w = max_h = 0;
       if ((frame_number_array) && (frame_pool)) {
	  for (int i=0 ; i<num_frames ; i++) {
	     frame = frame_number_array[i];	     
	     w = frame_pool->getFrame(frame)->width();
	     h = frame_pool->getFrame(frame)->height();
	     if (w > max_w) max_w = w;
	     if (h > max_h) max_h = h;
	  }
	  max_width = max_w;
	  max_height = max_h;
       } else ww_FATAL_ERROR("C_FrameList::getMaxBounds(short&,short&) failed! Tables not specified!\n");
    }
    bool allocFrameList ( short num ) {
       short *temp;
       if (num <= 0) ww_FATAL_ERROR("C_FrameList::allocFrameList(short) failed! (num <= 0)\n");
       temp = new short[num];
       if (temp) {
	  setNumFrames(num);
	  setFrameNumberArray(temp);
	  return TRUE;
       }
       return FALSE;
    }
    void destroyFrameList() {
       if (frame_number_array)
	 delete frame_number_array;
       
       frame_number_array = NULL;
    }
   
    C_FrameList() {
       max_width = max_height = 0; 
       setNumFrames(0);
       setFramePool(NULL);
       setFrameNumberArray(NULL);      
    }
    ~C_FrameList() {
       destroyFrameList();
    }
};
///////////////////////////////////////////////////////////////////////
// Class:     C_Sprite
// Purpose:   A sprite encloses an array of frame lists from which
//            frame lists and from which frames are selected and
//            displayed in the order determined by the program's
//            animation logic. (a decision based on the state of the
//            entity etc.)
// NOTE:      The aforementioned animation logic may be implemented
//            by inheriting this class and adding routines for the
//            logic or by using external code in conjunction with the
//            services provided by this class.
///////////////////////////////////////////////////////////////////////
#define csp_DEFAULT_SAVEDBG_BUFFER_WIDTH  100
#define csp_DEFAULT_SAVEDBG_BUFFER_HEIGHT 100
class C_Sprite {
    protected:
// Attached sprites are handy, for example, when we want to animate an
// explosion on top of a tank. We just 'attach' the explosion sprite to
// the tank sprite and off we go.
    C_Sprite      *attached;
    C_Sprite      *to_which_attached;

// The following contains a pointer to an array containing frame lists.
// A frame list is a structure in which animation frame numbers are stored
// for each 'instance' of the sprite.     
    C_FrameList   *frame_list_array;
    
    short          current_frame_list;
    short          num_frame_lists;
    
    short          current_frame;
// Remember the position we drew to the last time.
    short          last_x0, last_y0;
    short          last_frame;
// This flag indicates that background restoration hasn't been performed
// since the last draw operation.
    bool           dirty_;
// Leave the background dirty?
    bool           dont_save_bg;
// Draw only once?
    bool           draw_once;
    bool           drawn_once;
// Draw at all?
    bool           dont_draw;
// Auto-animation?
    bool           auto_anim;
    // Methods
public:
    void           setAutoAnimation ( bool f ) { auto_anim = f; };
    bool           autoAnimation() { return (auto_anim); };
protected:
// 'Autonomous sprites'
// Auto-destruct after the last frame? (This can sometimes be useful with attached
// sprites.) Used in conjunction with auto-animation.
    bool           destroy_after_last_frame;  
    // Methods
public:
    void           setDestroyAfterLastFrame ( bool f ) { destroy_after_last_frame = f; };
    bool           destroyAfterLastFrame() { return (destroy_after_last_frame); };
protected:
// Destroy this sprite after restoring background?
    bool           destroy_when_clean;
    // Methods
public:
    void           setDestroyWhenClean ( bool f ) { destroy_when_clean = f; };
    bool           destroyWhenClean() { return (destroy_when_clean); };
protected:
// Displacement values for attached sprites.
    short          x_offset;
    short          y_offset;
    // Methods (added 12/01/2000)
public:
    short          xOffset() { return (x_offset); };
    short          yOffset() { return (y_offset); };
    void           setXOffset ( short xo ) { x_offset = xo; };
    void           setYOffset ( short yo ) { y_offset = yo; };
    void           setXYOffsets ( short xo, short yo ) {
       setXOffset(xo);
       setYOffset(yo);
    }
protected:
// This pointer points to a bitmap that contains saved background data
// , so that we can restore the world bitmap when we need to redraw
// a sprite to some other location.
// This bitmap's dimensions must equal to the bounding box of all
// frames in the lists.
    C_Bitmap      *saved_bg; 
    
    public:    
    C_FrameList *frameListPtr( short flist ) { return(&frame_list_array[flist]); };
    short numFrameLists() { return (num_frame_lists); };    
    short frame() { return (current_frame); };
    short frameList() { return (current_frame_list); };
    void frameBounds ( short &w, short &h ) {
       C_Frame *this_frame;
       this_frame = frameListPtr(frameList())->getFrame(frame());
       w = this_frame->width();
       h = this_frame->height();
    }
    short nextFrame() {
       short num_frames = frameListPtr(frameList())->numFrames();    
       return ((frame()+1) % num_frames);
    }
    void  animate() {       
       setFrame(nextFrame());
       
       if (destroyAfterLastFrame()) {
	  // If the frame count wraps we know it's time to go...
	  if (frame() == 0) {
            detachThis();	     
 	    delete this;
	  }
       }
    }
    short lastX0() { return (last_x0); };
    short lastY0() { return (last_y0); };
    short lastFrame() { return (last_frame); };
    bool  dirty() { return (dirty_); };
    bool  dontSaveBackground() { return (dont_save_bg); };
    bool  dontDraw() { return (dont_draw); };   
    bool  drawOnce() { return (draw_once); };
    C_Bitmap *savedBackground() { return (saved_bg); };
    bool isValidFrameList ( short frame_list ) {
	return ((frame_list >= 0) & (frame_list < num_frame_lists));
    }
    void setFrameListArray ( C_FrameList *a ) { frame_list_array = a; };
    void setNumFrameLists ( short num ) { num_frame_lists = num; };
    void setFrameList ( short frame_list ) { current_frame_list = frame_list; };
    void setFrame ( short frame ) { current_frame = frame; };
    void setLastX0 ( short x0 ) { last_x0 = x0; };
    void setLastY0 ( short y0 ) { last_y0 = y0; };
    void setLastFrame ( short frame ) { last_frame = frame; };
    void setDirty ( bool d ) { dirty_ = d; };
    void setDontDraw ( bool f ) { dont_draw = f; };
    void setDrawOnce ( bool f ) {
       draw_once = f;              
    }    
    void setDontSaveBackground ( bool f ) { 
       dont_save_bg = f;
       if (dont_save_bg) {
	  if (saved_bg) {
	     delete saved_bg;
	     saved_bg = NULL;
	     setDirty(FALSE);
	  } 
       } else {
	  if (!saved_bg) setSavedBackgroundBuffer(csp_DEFAULT_SAVEDBG_BUFFER_WIDTH,
						  csp_DEFAULT_SAVEDBG_BUFFER_HEIGHT);
       }
    }
    void setSavedBackgroundBuffer ( short w, short h ) {
       if (saved_bg)
	 delete saved_bg;
       if ((w * h) > 0) 
	 saved_bg = new C_Bitmap(w, h, -1);
       else saved_bg = NULL;
    }
    void optimizeBackgroundBufferSize() {
       short max_w=0, max_h=0;
       short w, h;
       if (frame_list_array) {
	  for (int i=0 ; i<num_frame_lists ; i++) {
  	     frame_list_array[i].getMaxBounds(w, h);
	     if (w > max_w) max_w = w;
	     if (h > max_h) max_h = h;
	  }
  	  setSavedBackgroundBuffer(max_w,max_h);	  
       } else ww_FATAL_ERROR("C_Sprite::optimizeBackgroundBuffer() failed! No framelistarray!\n");
    }
    C_Sprite *attachment() { return (attached); };
    C_Sprite *toWhichAttached() { return (to_which_attached); };       

    C_Sprite *duplicate() {
       C_Sprite *temp;
       temp = new C_Sprite;
       *temp = *this;
       if (saved_bg) 
          temp->setSavedBackgroundBuffer(saved_bg->width(), saved_bg->height());
       return temp;
    }
// Attaches a sprite at the end of the list.
    void attach ( C_Sprite *att ) {
       if (attachment()) {
	  attachment()->attach(att);
       } else {       
          attached = att;	  
	  att->to_which_attached = this;
       }       
    }
// Removes the last attachment in the list.
// Returns a pointer to the sprite that was detached.
    C_Sprite *detach ( void ) {
       C_Sprite *temp;
       if (attachment()->attachment()) {
	  return (attachment()->detach());
       } else {
	  temp = attached;
	  attached->to_which_attached = NULL;
	  attached = NULL;	  
       }
       return(temp);
    }
// Detach this sprite from other sprites in the list.
    void detachThis() {
       C_Sprite *prev, *next;       
       prev = to_which_attached;
       next = attached;
       if (prev) prev->attached = next;
       if (next) next->to_which_attached = prev;
       to_which_attached = attached = NULL;
    }
    

// Draws the current frame and saves the background for restoration purposes. 
  
    void drawAt ( short x0, short y0, C_DrawSurface *dest ) {
       C_Frame *this_frame;
       short tx, ty;
              
       if (!dontDraw()) { 
	  if ((!drawOnce()) || (!drawn_once)) {
	     setDirty(TRUE);
	     this_frame = frameListPtr(frameList())->getFrame(frame());
	     tx = x0 - this_frame->xOffset() + xOffset();
	     ty = y0 - this_frame->yOffset() + yOffset();
	     setLastX0(tx);
	     setLastY0(ty);
	     setLastFrame(frame());
	     // Allow 'delayed' background restoration.
	     // (e.g. when a static object, such as a building explodes
	     // its background needs to be restored.)
	     if (drawOnce()) 
	       setDirty(FALSE);
	     
	     if (!dontSaveBackground())
	       bitBlt(savedBackground(), 0, 0, dest, tx, ty, this_frame->width(), this_frame->height(), QWidget::CopyROP, TRUE);
	     
	     bitBlt(dest, tx, ty, this_frame, 0, 0, -1, -1, QWidget::CopyROP, FALSE);
	  }
	  drawn_once = TRUE;
       }
       // Allow regular attached sprites despite possible drawOnce() == TRUE.
       // (e.g. a rotating satellite dish on top of a building.)
       if (attachment()) {
	  attachment()->drawAt(x0, y0, dest);
       }              
    }
    void restoreBackground( C_DrawSurface *dest ) {        
        C_Frame  *this_frame;
        // Get to the last attachment and start traversing backwards.
        if (attachment()) 
	   attachment()->restoreBackground(dest);
         
        if ((dirty()) && (!dontSaveBackground())) {
	   this_frame = frameListPtr(frameList())->getFrame(lastFrame());
	   bitBlt(dest, lastX0(), lastY0(), savedBackground(), 0, 0, this_frame->width(), this_frame->height(), QWidget::CopyROP, TRUE);
           setDirty(FALSE);
	}      
        if (destroyWhenClean()) {
	   // Commit suicide.
	   detachThis();
	   delete this;
	}
        if (auto_anim)
	   animate();
    }
// Pasting means that no background data is saved. So, it's permanent.
// This is ideal for drawing pock (or even tread) marks onto the world
// bitmap.
// NOTE: DOES NOT support attached sprites!!!
    void paste ( short x0, short y0, C_DrawSurface *dest ) {
       C_Frame *this_frame;     
       this_frame = frameListPtr(frameList())->getFrame(frame());
       bitBlt(dest, x0 - this_frame->xOffset(), y0 - this_frame->yOffset(), this_frame, 0, 0, -1, -1, QWidget::CopyROP, FALSE);
    }   

    C_Sprite() {
       drawn_once = FALSE;
       saved_bg = NULL;
       dont_save_bg = FALSE;
       attached = NULL;  
       to_which_attached = NULL;
       setDirty(FALSE);
       setNumFrameLists(0);
       setFrameListArray(NULL);
       setFrameList(0);
       setFrame(0);
       setDrawOnce(FALSE);
       setDontDraw(FALSE);
       setDestroyWhenClean(FALSE);
       setAutoAnimation(FALSE);
       setDestroyAfterLastFrame(FALSE);
       
       setXYOffsets(0,0);
       setSavedBackgroundBuffer(csp_DEFAULT_SAVEDBG_BUFFER_WIDTH,
	 		        csp_DEFAULT_SAVEDBG_BUFFER_HEIGHT);
    }
    ~C_Sprite() {       
       if (attached) attached->to_which_attached = NULL;
       if (to_which_attached) to_which_attached->attached = NULL;
       if (saved_bg) 
	  delete saved_bg;	      
       saved_bg = NULL;
    }
};



////////////////////////////////////////////////////////////////////////
// Class:     C_Player
// Purpose:   Stores information about a player, his name, his kills,
//            his everything.
////////////////////////////////////////////////////////////////////////
typedef enum {
   pc_RED = 0,
   pc_YELLOW = 1,
   pc_BLUE = 2,
   pc_GREEN = 3,
   pc_GRAY = 4,
   pc_DEFAULT = pc_RED
} PLAYERCOLOR;

class C_Player {
    protected:
    char        *nick_;
    void        *unit_list_;
    short        kills_;
    short        losses_;
    PLAYERCOLOR  color_;
    float        resources_;  
 
    public:

// Constructors
    void setNickName ( char *nickname ) {
        nick_ = new char[strlen(nickname)+1];
        strcpy(nick_, nickname);
    }
    void setUnitList ( void *list ) { unit_list_ = list; };
    void setKills ( short k ) { kills_ = k; };
    void setLosses ( short l ) { losses_ = l; };
    void setResources ( float r ) { resources_ = r; }; 
    void setColor ( PLAYERCOLOR c ) { color_ = c; };
    void *unitList() { return (unit_list_); };
    char *nickName() { return (nick_); };
    short kills() { return (kills_); };
    short losses() { return (losses_); };
    float resources() { return (resources_); };
    PLAYERCOLOR color() { return (color_); };
   
    C_Player ( char * nickname, void * unit_list, PLAYERCOLOR c ) {
        setNickName(nickname);
        setUnitList(unit_list);       
        setColor(c);
        setKills(0);       
        setLosses(0);
        setResources(0.0);
    }
    C_Player ( char * nickname, PLAYERCOLOR c ) {	
	setNickName(nickname);
        setKills(0);
        setLosses(0);
        setColor(c);
        setUnitList(NULL);
        setResources(0.0);
    }
    C_Player() {
        setNickName("NONE");
        setKills(0);
        setLosses(0);
        setColor(pc_DEFAULT);
        setUnitList(NULL);
        setResources(0.0);
    }
    
};

/////////////////////////////////////////////////////////////////////////
// Class:      C_Unit
// Purpose:    Contains status information for a single unit in the 
//             field.
/////////////////////////////////////////////////////////////////////////

class C_Unit {
    protected:    
// Global ID for this unit (used by both the server and the client)
    short        unit_ID;  
// All units in the world are kept in a doubly linked list.
    C_Unit      *next_unit;
    C_Unit      *prev_unit;
// The graphic image representing this unit in the battle field.
    C_Sprite    *face;
// A link to the player that owns this particular unit.
    short        owner_player_ID_;
//    C_Player    *owner_;
// Unit type parameters specify the physical characteristics of this unit.
    short        type_ID;
    bool         is_selected;
    bool         do_not_draw;
    bool         is_disabled;
//    C_UnitParam *unit_type_data;
// The current 'hull' value indicates how much energy is left.
    short        hull_;
// NOTE: The following data are manipulated as needed by the server software.
// Heading.
   float         world_direction;
// Position in world-space.
   float         world_x, world_y;
// Yes, it's our speed.
   float         speed_;
// Waypoint information (extension 10/01/2000)
   short         waypoint_x, waypoint_y;
   short         state_;  // In what state are we john?.
// A timestamp value for general purposes
// i.e. does not necessarily represent the 'age' of 
// a unit.
    unsigned long time_stamp;
// Public methods 
public:  
    short state() { return (state_); };
    void  setState( short s ) { state_ = s; }; 
    short wayPointX() { return (waypoint_x); };
    short wayPointY() { return (waypoint_y); };
    void setWayPointX ( short x ) { waypoint_x = x; };
    void setWayPointY ( short y ) { waypoint_y = y; };
    void setWayPointXY ( short x, short y ) {
       setWayPointX(x);
       setWayPointY(y);
    }
    unsigned long timeStamp() { return (time_stamp); };
    void setTimeStamp ( unsigned long ts ) { time_stamp = ts; };
    bool doNotDraw() { return (do_not_draw); };
    void setDoNotDraw( bool f ) { do_not_draw = f; };
    bool isSelected() { return (is_selected); };
    bool isDisabled() { return (is_disabled); };
    void disable() { is_disabled = TRUE; };
    void enable() { is_disabled = FALSE; };
    void select() { is_selected = TRUE; };
    void unselect() { is_selected = FALSE; };
    short hull() { return (hull_); };
    bool  alive() { return (hull_ > 0); };
    short unitID() { return (unit_ID); };
    short typeID() { return (type_ID); };
    C_Sprite *sprite() { return (face); };
    void setSprite ( C_Sprite *s ) {
       if (face)
	 delete face;
       
       if (s)
	 face = s;
       else ww_FATAL_ERROR("C_Unit::setSprite() failed! (s == NULL)\n");
    } 
  //short heading() { return ((short)world_direction); };
    float heading() { return (world_direction); };
    float worldX() { return (world_x); };
    float worldY() { return (world_y); };
    float speed() { return (speed_); };
    void setSpeed ( float s ) { speed_ = s; };
    short ownerPlayerID() { return (owner_player_ID_); };
    C_Unit *nextUnit() { return (next_unit); };
    C_Unit *prevUnit() { return (prev_unit); };
   
    virtual C_Unit *duplicate() {      
       C_Unit *temp;
       temp = new C_Unit;
       *temp = *this;
       // Tell the destructor that deleting the sprite is a big no-no.
       // A new sprite inherits the pointer.
       preserveSprite();
       temp->next_unit = temp->prev_unit = NULL;
       return temp;
    }
  //void setHeading ( short hdg ) { world_direction = (float)hdg; };
    void setHeading ( float hdg ) { world_direction = hdg; };
    void setWorldX ( float x ) { world_x = x; };
    void setWorldY ( float y ) { world_y = y; };
    void setWorldXY ( float x, float y ) {
       setWorldX(x);
       setWorldY(y);
    }
    void setUnitID ( short id ) { unit_ID = id; };
    void setTypeID ( short id ) { type_ID = id; };
    void setHull ( short h ) {  hull_ = h; };
    void setOwnerPlayerID ( short own ) { owner_player_ID_ = own; };
    void setNextUnit ( C_Unit *unit ) { next_unit = unit; };
    void setPrevUnit ( C_Unit *unit ) { prev_unit = unit; };
 
 protected:
    bool preserve_flag;
    void preserveSprite() { preserve_flag = TRUE; };
 public: 
    //----------------------------------------------------------------------------------
    virtual void animateUnit() { printf("C_Unit::animateUnit() (virtual) abused!\n"); };
    virtual void getBounds( short &, short & ) { printf("C_Unit::getBounds() (virtual) abused!\n"); };   
    virtual void handleStates() { printf("C_Unit::handleStates() (virtual) abused!\n"); };
    virtual void destroyUnit() { printf("C_Unit::destroyUnit() (virtual) abused!\n"); };
    virtual void spawnUnit() { printf("C_Unit::spawnUnit() (virtual) abused!\n"); };
    virtual void hitUnit( short ) { printf("C_Unit::hitUnit() (virtual) abused!\n"); };
    virtual void unitShoot() { printf("C_Unit::unitShoot() (virtual) abused!\n"); };
    virtual void handleTranslation() { printf("C_Unit::handleTranslation() (virtual) abused!\n"); };
    //----------------------------------------------------------------------------------  
    C_Unit() {
       is_selected = FALSE;
       is_disabled = FALSE;
       setDoNotDraw(FALSE);
       setOwnerPlayerID(0);
       setNextUnit(NULL);
       setPrevUnit(NULL);
       setWayPointXY(500,500);
       setTimeStamp(0);       
       face = NULL;
       preserve_flag = FALSE;       
    }   
   
   virtual ~C_Unit() {
      if (!preserve_flag) {
         if (face) delete face;
      }
   }
};

#include "wwtypeid.cc"
// This is the hard-coded maximum number of players.
#define ww_MAX_PLAYERS     4
#define ww_PLAYERID_RED    0
#define ww_PLAYERID_GREEN  1
#define ww_PLAYERID_BLUE   2
#define ww_PLAYERID_YELLOW 3
#define ww_PLAYERID_GOD    0xFFFF
// The upper half of these is reserved for 'stunt doubles'.
#define ww_MAX_TOTAL_UNITS 2000
static bool unitIDBoundsCheck ( short unit_id ) {
  if ((unit_id >= 0) && (unit_id < ww_MAX_TOTAL_UNITS))
    return TRUE;
  return FALSE;
}
// Define battle field characteristics.
class C_BattleField {
 protected:
// 'player ID' is used as the index with this sucker.
   C_Player  players_on_the_field[ww_MAX_PLAYERS];
// 'Unit ID' is the index for this one. Not hard.
// We want to create and destroy units dynamically.
   C_Unit  *units_on_the_field[ww_MAX_TOTAL_UNITS];
   short    first_used_unit_id;
   short    last_used_unit_id;
// 'Type ID'
  //   C_UnitParam unit_type_data[ww_NUM_UNIT_TYPES];
   
   int       player_mouse_x[ww_MAX_PLAYERS];
   int       player_mouse_y[ww_MAX_PLAYERS];
   
   short     num_players;
   short     num_units;
   short     num_units_per_player[ww_MAX_PLAYERS];
      
   short     map_width, map_height;
   int       map_seed;
   // Update unit-2-unit links as needed depending on
   // whether we are allocating a new unit or destroying
   // an old one.
   void updateUnitLinks ( short unit_id, bool do_insert ) {
      C_Unit *next, *prev;
      short i;
      if ((unit_id >= 0) && (unit_id < ww_MAX_TOTAL_UNITS)) {
	 if (do_insert) { // insertion
	    for (i=unit_id-1 ; i >= 0 ; i--) {
	       if (unitExists(i)) {
		  i--;
		  break;
	       }
	    }
	    i++;
	    if (unit_id == 0) prev = NULL; 
	    else prev = units_on_the_field[i];
	 
	    for (i=unit_id+1 ; i<ww_MAX_TOTAL_UNITS ; i++) {
	       if (unitExists(i)) {
		  i++;
		  break;
	       }
	    }
	    i--;
	    if (unit_id == ww_MAX_TOTAL_UNITS-1) next = NULL;
	    else next = units_on_the_field[i];
	  
	    // Check whether we are inserting a unit or removing one.
	    //if (unitExists(unit_id)) {
	    // Inserting one..
	    if (unit_id < first_used_unit_id) 
	      first_used_unit_id = unit_id;
	    if (unit_id > last_used_unit_id)
	      last_used_unit_id = unit_id;
	    
	    unit(unit_id)->setPrevUnit(prev);
	    unit(unit_id)->setNextUnit(next);
	    if (prev) 
	      prev->setNextUnit(unit(unit_id));
	    if (next)
	      next->setPrevUnit(unit(unit_id));
	    
	 } else {
	    // Removal
	    prev = units_on_the_field[unit_id]->prevUnit();
	    next = units_on_the_field[unit_id]->nextUnit();
	    
	    if (unit_id == first_used_unit_id) {
	       if (next) 
       		  first_used_unit_id = next->unitID();
	       else first_used_unit_id = ww_MAX_TOTAL_UNITS-1;
	    }
	    if (unit_id == last_used_unit_id) {
	       if (prev)
		 last_used_unit_id = prev->unitID();
	       else last_used_unit_id = 0;
	    }
	    if (prev) {
	       prev->setNextUnit(next);	       
	    }
	    if (next) {
	       next->setPrevUnit(prev);
	    }
	 }
      } else ww_FATAL_ERROR("C_BattleField::updateUnitLinks() failed! unit_id out of bounds!\n");
   }
 public:
   C_Unit *firstExistingUnit() {
      if (num_units > 0) 
	return (unit(first_used_unit_id));
      return NULL;
   }
   C_Unit *lastExistingUnit() {
      if (num_units > 0)
	return (unit(last_used_unit_id));
      return NULL;
   }
   void allocUnit ( short unit_id, C_Unit *unit ) {
      if ((unit_id >= 0) && (unit_id < ww_MAX_TOTAL_UNITS)) {
	 // If a unit with this id already exists we're in trouble..
	 if (unitExists(unit_id)) {
	    printf("C_BattleField::allocUnit(short): realloc at unit_id = %i!\n",unit_id);
            destroyUnit(unit_id); 	    
	 }
         if (!unit) {
	    ww_FATAL_ERROR("C_BattleField::allocUnit() failed! (unit == NULL)\n");
	 }
	 units_on_the_field[unit_id] = unit; //new C_Unit;
         units_on_the_field[unit_id]->setUnitID(unit_id);
         // Update linked lists
	 updateUnitLinks(unit_id, TRUE);
	 num_units++;
	 if (!unitExists(unit_id))
 	    ww_FATAL_ERROR("C_BattleField::allocUnit(short) failed! new allocation failed!\n");
      } else ww_FATAL_ERROR("C_BattleField::allocUnit(short) failed! unit_id out of bounds!\n");
   }
   void destroyUnit ( short unit_id ) {
      if ((unit_id >= 0) && (unit_id < ww_MAX_TOTAL_UNITS)) {
	 if (units_on_the_field[unit_id]) {
	    updateUnitLinks(unit_id, FALSE);
       	    delete units_on_the_field[unit_id];
	    units_on_the_field[unit_id] = NULL;	              
	    num_units--;
	 }
      } else ww_FATAL_ERROR("C_BattleField::destroyUnit(short) failed! unit_id out of bounds!\n");
   }
   void setNumUnitsPerPlayer ( short p, short n ) {
      if ((p >= 0) && (p < num_players)) {
         num_units_per_player[p] = n; 
      } else ww_FATAL_ERROR("C_BattleField::setNumUnitsPerPlayer(short,short) failed! player_id out of bounds!\n");
   };
   short numUnitsPerPlayer( short p ) {
      if ((p >= 0) && (p < num_players)) 
         return (num_units_per_player[p]); 
      ww_FATAL_ERROR("C_BattleField::numUnitsPerPlayer(short) failed! player_id out of bounds!\n");
   };
   void setPlayerMouseX ( short p, short x ) {
      if ((p >= 0) && (p < num_players)) 
	player_mouse_x[p] = x;
   }
   void setPlayerMouseY ( short p, short y ) {
      if ((p >= 0) && (p < num_players))
	player_mouse_y[p] = y;
   }
   void setPlayerMouseXY ( short p, short x, short y ) {
      if ((p >= 0) && (p < num_players)) {
	 setPlayerMouseX(p, x);
	 setPlayerMouseY(p, y);
      }
   }
   short playerMouseX ( short p ) {
      if ((p >= 0) && (p < num_players)) 
         return (player_mouse_x[p]);
      else return 0;      
   }
   short playerMouseY ( short p ) {
      if ((p >= 0) && (p < num_players)) 
	return (player_mouse_y[p]);
      else return 0;
   }
   
   C_Player *player ( short player_id ) {
      if ((player_id >= 0) && (player_id < num_players))
         return (&players_on_the_field[player_id]);
      ww_FATAL_ERROR("C_BattleField::*player(short) failed! player_id out of bounds!\n");
   }
   bool unitExists ( short unit_id ) {
      if ((unit_id >= 0) && (unit_id < ww_MAX_TOTAL_UNITS)) {
	 return (units_on_the_field[unit_id] != NULL);
      }
      ww_FATAL_ERROR("C_BattleField::unitExists(short) failed! unit_id out of bounds!\n");
   }
   C_Unit *unit ( short unit_id ) {
      if ((unit_id >= 0) && (unit_id < ww_MAX_TOTAL_UNITS))
	return (units_on_the_field[unit_id]);
      ww_FATAL_ERROR("C_BattleField::*unit(short) failed! unit_id out of bounds!\n");
   }
#if 0
   // Deep copy.
   void copyUnit ( short old_id, short new_id ) {     
      if (unitExists(old_id)) {
	 if (unitExists(new_id)) {
	    printf("C_BattleField::copyUnit(): WARN: copied an unit to an occupied slot!\n");
	    destroyUnit(new_id);	    
	 }
         units_on_the_field[new_id] = unit(old_id)->duplicate();
	 units_on_the_field[new_id]->setUnitID(new_id);
	 updateUnitLinks(new_id, TRUE);
	 num_units++;
      } else ww_FATAL_ERROR("C_BattleField::copyUnit() failed! Source unit does not exist!\n");
   }
#endif 
   void moveUnit ( short old_id, short new_id ) {
      C_Unit *unit_;
      if (unitExists(old_id)) {
	 if (unitExists(new_id)) {
	    printf("C_BattleField::moveUnit() WARN: moved to an already occupied slot!\n");
	 }
         unit_ = unit(old_id);
         updateUnitLinks(old_id, FALSE);
         units_on_the_field[old_id] = NULL;
         num_units--;
        
         units_on_the_field[new_id] = unit_;
	 units_on_the_field[new_id]->setUnitID(new_id);
         updateUnitLinks(new_id, TRUE);
         num_units++;
	               
      } else ww_FATAL_ERROR("C_BattleField::moveUnit() failed! source unit does not exist!\n");
      
   }
  /*
   C_UnitParam *typeData ( short type_id ) {
      if ((type_id >= 0) && (type_id < ww_NUM_UNIT_TYPES))
	 return (&unit_type_data[type_id]);
      ww_FATAL_ERROR("C_BattleField::*typeData(short) failed! type_id out of bounds!\n");
      }*/
   //C_Unit *unitsOnTheField() { return (units_on_the_field); };
   short numUnits() { return (num_units); };
   short numPlayers() { return (num_players); };
   short mapWidth() { return (map_width); };
   short mapHeight() { return (map_height); };
   int   mapSeed() { return (map_seed); };
   void  setNumUnits ( short n ) {
      if (n > ww_MAX_TOTAL_UNITS-1) ww_FATAL_ERROR("C_BattleField::setNumUnits() failed! Increment ww_MAX_TOTAL_UNITS and recompile!\n");
      num_units = n;
   }
   void  setNumPlayers ( short n ) {
      if (n > ww_MAX_PLAYERS) ww_FATAL_ERROR("C_BattleField::setNumPlayers() failed! Increment ww_MAX_PLAYERS and recompile.\n");
      num_players = n;
   }
   void  setMapWidth ( short mw ) { map_width = mw; };
   void  setMapHeight ( short mh ) { map_height = mh; };
   void  setMapSeed ( short s ) { map_seed = s; };
   
   C_BattleField() {
      setNumPlayers(0);
      setNumUnits(0);
      first_used_unit_id = ww_MAX_TOTAL_UNITS-1;
      last_used_unit_id = 0;
      for (int i=0 ; i<ww_MAX_TOTAL_UNITS ; i++) 
	units_on_the_field[i] = NULL;
   }
   
};


#endif

