#ifndef __WWUNITDEFS_CC_INCLUDED__
#define __WWUNITDEFS_CC_INCLUDED__
//========================================
// Unit class definitions for WarWorld
// Written by Jani Argillander
//            (japear@utu.fi)
// January 2000
//========================================
#include "wwdefs.cc"

#define ww_STATE_NORMAL   0
#define ww_STATE_SPAWN    1
#define ww_STATE_EXPLODE  2
#define ww_STATE_TAKE_HIT 3
#define ww_STATE_SHOOT    4

class WWAnimatedUnit : public C_Unit {
protected:
 
  //====================================================================================================
  // Would you prefer more virtuals, sir? 
  virtual void attachExplosionAnimation() {
    WWExplosionSprite *new_s = new WWExplosionSprite;
    sprite()->attach(new_s);
  }
  virtual void animateExplosion() {};
  virtual void detachExplosionAnimation() {};
  virtual void cleanupForDestruction() {};
  virtual void attachSpawnAnimation() {
    WWSpawnSprite *new_s = new WWSpawnSprite;
    sprite()->attach(new_s);
  }
  virtual void animateSpawn() {};
  virtual void detachSpawnAnimation() {};
  virtual void attachHitAnimation() {
    WWHitSprite *new_s;
    sprite()->attach(new_s);
  }
  virtual void detachHitAnimation() {};
  virtual void attachShotAnimation() {};  
  virtual void detachShotAnimation() {};  
  virtual void drawCraters() {  // A big unit leaves great craters! 
    gfx_crater_framelist.getFrame(0)->paste(windowDrawSurface, (short)worldX(), (short)worldY());
  }      

  virtual float acceleration() {
    return (0.0);
  }     
  virtual float turnRate() {
    return (0.0);
  }  
  virtual float maxSpeed() {
    return (0.0);
  }
public:
  virtual void getBounds ( short &w, short &h ) { printf("WARN: WWAnimatedUnit::getBounds() (virtual) abused!\n"); };
  // Do general animation stuff. 
  // (i.e. show the right frame at the right time.)
  virtual void animateUnit() { 
    printf("WARN: WWAnimatedUnit::animateUnit() (virtual) abused!\n"); 
  }
  // Enough virtuality already..
  void destroyUnit() {
    C_BattleField *b_field = battleField();
    C_Sprite *unitSelector;    
    int i;

    if (state() != ww_STATE_EXPLODE) {	 
      setHull(0);
      // Get rid of the unit selector sprite
      if (isSelected()) {
	unselect();	    
	if (ownerPlayerID() == thisPlayerID()) {
          unitSelector = sprite()->attachment();
          while(unitSelector->attachment()) unitSelector = unitSelector->attachment();	     
	  //while(unitSelector->attachment())      
	  //  unitSelector = sprite()->detach();
          
          unitSelector->setDestroyWhenClean(TRUE);
	  //unitSelector->restoreBackground(windowDrawSurface);
	  decrementNumUnitsSelected();
          
	  //delete unitSelector;
	}
      }
      // Find a free slot from the upper half of the world unit list.
      // The upper half is used to hold zombies. 	 
      for (i=ww_MAX_TOTAL_UNITS-1 ; i >= ww_MAX_TOTAL_UNITS/2 ; i--) {
	if (!b_field->unitExists(i)) {
	  // Make a 'stunt double' and destroy the original
	  // to make room for new units.
	  b_field->moveUnit(unitID(), i);

          attachExplosionAnimation();
	  //explode = new WWExplosionSprite;
	  //b_field->unit(i)->sprite()->attach(explode);
   
	  setState(ww_STATE_EXPLODE);
	  setTimeStamp(gameTicks());
	  disable();
	  i=0;
	  break;
	}
      }
      // If we couldn't get a stunt double, let's destroy the unit
      // without an explosion. It just disappears.
      if (i != 0) {
	// Leave gracefully.. this can cause some
	// visual artifacts. Not very often.
	sprite()->restoreBackground(windowDrawSurface);
	b_field->destroyUnit(unitID());   
      }
    }  
  }

  void spawnUnit() {
    C_Sprite *tmp;     
    tmp = sprite();
    while (tmp) {
      tmp->setDontDraw(TRUE);
      tmp = tmp->attachment();
    }
    attachSpawnAnimation();
    //tmp = new WWSpawnSprite;          
    //sprite()->attach(tmp);      
    // Prevent the owner from selecting this unit
    // while it is spawning.      
    setState(ww_STATE_SPAWN);
    setTimeStamp(gameTicks());
    disable();      
  }
  void hitUnit( short energy_remaining ) {
    //C_Sprite *temp;      
    //WWHitSprite *new_s = new WWHitSprite;      
    if (state() == ww_STATE_NORMAL) {
      setHull(energy_remaining);
      setState(ww_STATE_TAKE_HIT);
      setTimeStamp(gameTicks()); 
    }
  }
  void unitShoot() {
    if ((state() != ww_STATE_SPAWN) && (state() != ww_STATE_EXPLODE) && (unitIsVisible(unitID()))) {
      setState(ww_STATE_SHOOT); 
      setTimeStamp(gameTicks());  
    }
  }   
  virtual void handleDestruction() {
    C_BattleField *b_field = battleField();
    unsigned long time_elapsed = gameTicks() - timeStamp();
          
    // Control the explosion
    if (isSelected()) unselect();
   
    // The larger the better..
    if (time_elapsed == 4) {
      sprite()->setDontDraw(TRUE);
      sprite()->setDontSaveBackground(TRUE);
     
      drawCraters();
      // detach all sprites except the explosion and the first sprite
      // in the list (this prevents us from seeing craters
      // with rotating gun barrels...:)
      cleanupForDestruction();
    }	 
    // Get rid of the unit completely
    if (time_elapsed >= 8) {   
      detachExplosionAnimation();
      b_field->destroyUnit(unitID());
      return;
    }
    animateExplosion();	 	    
  }
  virtual void handleSpawning() {
    C_Sprite *tmp;
    unsigned long time_elapsed = gameTicks() - timeStamp(); 
    // As the cloud of confusion gets bigger
    // put the sprite for the unit under
    // that cloud..
    if (time_elapsed == 5) {  
      tmp = sprite();
      while (tmp) {
	tmp->setDontDraw(FALSE);
	tmp = tmp->attachment();
      }           		  
    }
    // After the last frame get rid of the cloud.
    if (time_elapsed >= 8) { 	    
      enable();
      setState(ww_STATE_NORMAL);
      setTimeStamp(gameTicks());
      detachSpawnAnimation();    
    }
    // Next frame!
    animateSpawn();	      
  }

  virtual void handleHits() {
    C_Sprite *temp;
    if ((isSelected()) && (ownerPlayerID() == thisPlayerID())) {
      temp = sprite()->detach();
      attachHitAnimation();	 
      sprite()->attach(temp);
    } else {
      attachHitAnimation();
    }
    setState(ww_STATE_NORMAL);
  }
  virtual void handleShots() {
    C_Sprite *temp;
    if ((isSelected()) && (ownerPlayerID() == thisPlayerID())) {
      temp = sprite()->detach();
      attachShotAnimation();	 
      sprite()->attach(temp);
    } else {
      attachShotAnimation();
    }
    setState(ww_STATE_NORMAL);
  }

 
  float tnt_waypoint_x, tnt_waypoint_y;
  float tnt_unit_trans_vec_i, tnt_unit_trans_vec_j;
  float tnt_waypoint_bearing;

  void turn_and_translate() {
    const float epsilon = (256.0/16.0)/2.0;
    float head;
    float delta;
    float turn;
    float new_head;    

    float distance;
    float x_dif, y_dif;

    if ((tnt_waypoint_x != wayPointX()) || (tnt_waypoint_y != wayPointY())) {
      // Set current waypoint
      tnt_waypoint_x = wayPointX();
      tnt_waypoint_y = wayPointY();
      // Waypoint changed! Recalculate.
      x_dif = tnt_waypoint_x - worldX();     // difference between the current location and the new waypoint. 
      y_dif = tnt_waypoint_y - worldY();
      // Get bearing
      tnt_waypoint_bearing = getAngle256ForVector(x_dif, -y_dif);
      // How far are we supposed to drive?
      distance = sqrt(x_dif*x_dif + y_dif*y_dif);
      tnt_unit_trans_vec_i = x_dif / distance;
      tnt_unit_trans_vec_j = y_dif / distance;
    }
    // Are we there already?
    if ((worldX() == tnt_waypoint_x) && (worldY() == tnt_waypoint_y))
      return;

    // Temporary  
    head = heading();

    // Check whether heading corrections are needed. 
    if (((short)head > ((short)tnt_waypoint_bearing - (short)epsilon)) && ((short)head < ((short)tnt_waypoint_bearing + (short)epsilon))) {
      // Nope.
      translate(tnt_unit_trans_vec_i, tnt_unit_trans_vec_j);  
      setHeading((short)tnt_waypoint_bearing);
    } else {
      // Sure.
      turn = DEG_TO_ANGLE256(turnRate()) /(float)frameRate();

      delta = tnt_waypoint_bearing - head;
      // Turning AI :)
      // Take a left or right turn depending on which is a smart thing to do.
      if (delta < -0.00001) {
        if (-delta < DEG_TO_ANGLE256(180.0)) {
	  turn = -turn; 
        } else turn = turn;
      } else {
        if (delta < DEG_TO_ANGLE256(180.0)) {
          turn = turn;
	} else turn = -turn;
      }
      // Turn
      new_head = head + turn;
      // Make sure the angle is between 0 and 255.
      if (new_head < -0.00001)
	new_head = 255.0 + new_head;
      if (new_head > 255.0)
	new_head = new_head - 255.0;
      // Set heading
      setHeading(new_head);
    } 
  }
  void translate ( float trans_vec_i, float trans_vec_j ) {
    float v = speed() + acceleration() / (float)frameRate();
  
    setSpeed((v < maxSpeed()) ? v : maxSpeed());
    setWorldXY(worldX() + trans_vec_i * speed() / (float)frameRate(), worldY() + trans_vec_j * speed() / (float)frameRate()); 
    
    if (((short)worldX() > (short)((float)wayPointX() - trans_vec_i * 8.0  / 2.0))
      && ((short)worldX() < (short)((float)wayPointX() + trans_vec_i * 8.0 / 2.0))) {
      if (((short)worldY() > (short)((float)wayPointY() - trans_vec_j *8.0/ 2.0))
	&& ((short)worldY() < (short)((float)wayPointY() + trans_vec_j *8.0/ 2.0))) {
	setSpeed(0.0);
        setWorldXY((float)wayPointX(), (float)wayPointY());
      } 
    }
  }
  // Handle waypoints 
  virtual void handleTranslation() {
    //************************************************
    // WAYPOINTS DISABLED (NOT EVEN FULLY IMPLEMENTED)
    //************************************************
    //turn_and_translate();
  }  

  virtual void handleStates() {
    switch (state()) {
    case ww_STATE_SPAWN:
      handleSpawning();
      break;
    case ww_STATE_EXPLODE:
      handleDestruction();
      break;
    case ww_STATE_SHOOT:
      handleShots();
      handleTranslation();
      break;
    case ww_STATE_TAKE_HIT:
      handleHits();
      handleTranslation();
      break; 
    case ww_STATE_NORMAL:
      handleTranslation();
      break;
    }
  }
   
public:
  // Constructor(s)
  WWAnimatedUnit() : C_Unit() {
    setState(ww_STATE_NORMAL);
    tnt_waypoint_x = tnt_waypoint_y = -10000.0;    
  }
};




//============================================================================

class WWTank : public WWAnimatedUnit {
 private:
  // Translation-related constants
  const static float max_speed    = 20.0;    // m/s
  const static float acceleration_= 5.0;     // m/(s*s)
  const static float turn_rate    = 45.0;    // deg/s
  // Unit dimensions
  const static short bound_width  = 40;       
  const static short bound_height = 40;       
  
 public:
   float turnRate() { return (turn_rate); };
   float acceleration() { return (acceleration_); };
   float maxSpeed() { return (max_speed); };

   void getBounds ( short &w, short &h ) {
      w = bound_width;
      h = bound_height;
   }
   void animateUnit() {
      C_BattleField *b_field = battleField();      
      C_Sprite *temp = sprite();
      short owner;
      int unit_x, unit_y;
      int x_dif, y_dif;
        
      // Assume heading() e Int[0..255]
      temp->setFrameList((((255 - (short)heading()) >> 4) + 5) % 16);

      // If this tank is selected by its owner rotate the gun
      // according to his mouse cursor position. wow.
      if (isSelected()) {	      	 
	  owner = ownerPlayerID();
          unit_x = (int)worldX();
          unit_y = (int)worldY();
          x_dif =  b_field->playerMouseX(owner) - unit_x;
          y_dif =  b_field->playerMouseY(owner) - unit_y;
	  temp->attachment()->setFrameList((((short)getAngle256ForVector(x_dif, y_dif)+64+5) % 256) >> 4);
      } 
   }
  void attachExplosionAnimation() {
    WWExplosionSprite *new_s = new WWExplosionSprite;
    sprite()->attach(new_s);
  }
  void animateExplosion() { return; };
  // get rid of unnecessary sprites.
  void cleanupForDestruction() {
    C_Sprite *temp;
    // destroy the gun.
    temp = sprite()->attachment();
    temp->detachThis();
    delete temp; 
  } 

  void attachSpawnAnimation() {
    WWSpawnSprite *new_s = new WWSpawnSprite;
    sprite()->attach(new_s);
  }
  void animateSpawn() { return; };
 
  void attachHitAnimation() {
    WWHitSprite *new_s = new WWHitSprite;
    sprite()->attach(new_s);
  }
  void attachShotAnimation() {
    WWFlameSprite *new_s = new WWFlameSprite;
    float angle;
    
    sprite()->attach(new_s);
    new_s->setFrameList(sprite()->attachment()->frameList());
    angle = (float)new_s->frameList() * 2*3.14159/16.0;
    angle = 2*3.14159 - angle + 3.14159/2.0; 
    new_s->setXYOffsets((short)((float)20.0*cos(angle)), (short)((float)20.0*-sin(angle)));
  }
 
  WWTank ( short unit_id, short owner_id, short wx, short wy, short heading ) {
    setUnitID(unit_id);
    setTypeID(ww_TYPEID_TANK);
    setOwnerPlayerID(owner_id);
    setWorldXY(wx, wy);
    setHeading(heading);
    setHull(100);
    setState(ww_STATE_NORMAL);
  }
  WWTank() {
    setTypeID(ww_TYPEID_TANK);
    setHull(100);
  }
};

//=========================================================================================

class WWTankFactory : public WWAnimatedUnit {

public:
  float turnRate() { return (0.0); };
  float acceleration() { return (0.0); };
  float maxSpeed() { return (0.0); };

  void getBounds ( short &w, short &h ) {
    w = 40;
    h = 40;
  }
  void animateUnit() {};
  
  WWTankFactory ( short unit_id, short owner_id, short wx, short wy, short heading ) : WWAnimatedUnit() {
    setUnitID(unit_id);
    setTypeID(ww_TYPEID_TANK_FACTORY);
    setOwnerPlayerID(owner_id);
    setWorldXY(wx, wy);
    setHeading(heading);
    setHull(100);
  }
  WWTankFactory() : WWAnimatedUnit() {
    setTypeID(ww_TYPEID_TANK_FACTORY);
    setHull(100);
  }   
};

#endif




